/***************************************************************************
                          LOW_linkPassiveSerial.cpp  -  description
                             -------------------
    begin                : Sun Jul 7 2002
    copyright            : (C) 2002 by Harald Roelle, Helmut Reiser
    email                : roelle@informatik.uni-muenchen.de, reiser@informatik.uni-muenchen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <string>
#include <memory>


#include "LOW_linkPassiveSerial.h"

#include "LOW_platformMisc.h"
#include "LOW_IPCKeyGeneratorFactory.h"
#include "LOW_semaphoreSetFactory.h"



//=====================================================================================
//
// constructors
//

LOW_linkPassiveSerial::LOW_linkPassiveSerial( const LOW_portSerialFactory::portSpecifier_t &inSerPortSpec,
                                              const bool inAllowProgPulse) :
  LOW_link( false, false, false, inAllowProgPulse)
{
  serialPort   = LOW_portSerialFactory::new_portSerial( inSerPortSpec);

  auto_ptr<LOW_IPCKeyGenerator> keyGenerator (LOW_IPCKeyGeneratorFactory::new_IPCKeyGenerator());

  semSet       = LOW_semaphoreSetFactory::new_semaphoreSet( keyGenerator->getSemSetKey( inSerPortSpec), semaphoreCount, 1);
  
  resetLinkAdapter();
  resetBus(); // to detect a missing adapter
}

  

LOW_linkPassiveSerial::~LOW_linkPassiveSerial()
{
  serialPort->tty_flush();

  delete semSet;
  delete serialPort;
}



//=====================================================================================
//
// touch data methods
//

bool LOW_linkPassiveSerial::touchBit( const bool inSendBit, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  uint8_t         serSendByte;
  bool            retVal;

  serialPort->tty_flush();

  // get bit ready to be sent
  serSendByte = inSendBit ? 0xFF : 0x00;

  // Send the bit
  serialPort->tty_write( serSendByte);

  uint8_t tmpByte = serialPort->tty_readByte();        
  if ( tmpByte & 0x01 )
    retVal = true;
  else
    retVal = false;
  
  strongPullup( inPullup);
  
  return retVal;
}  
    

uint8_t LOW_linkPassiveSerial::touchByte( const uint8_t inSendByte, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  uint8_t  mask    = 0x01;
  uint8_t  retVal  = 0x00;
  
  for( int a=0; a<8; a++) {
    retVal |= touchBit( inSendByte & mask) ? mask : 0x00;
    mask <<= 1;
  }
  
  strongPullup( inPullup);
  
  return retVal;
}

  
byteVec_t LOW_linkPassiveSerial::touchBlock( const byteVec_t &inBytes, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  byteVec_t retVal = byteVec_t( inBytes.size());
  
  for( unsigned int i=0; i<inBytes.size(); i++) {
    retVal[i] = touchByte( inBytes[i], inPullup);
  }
  
  return retVal;
}


//=====================================================================================
//
// read data methods
//
  
bool LOW_linkPassiveSerial::readDataBit( const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  return touchBit( true, inPullup);
}

    
uint8_t LOW_linkPassiveSerial::readDataByte( const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  return touchByte( 0xff, inPullup);
}

    
void LOW_linkPassiveSerial::readData( byteVec_t &outBytes, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  for( unsigned int i=0; i<outBytes.size(); i++)
    outBytes[i] = readDataByte( inPullup);
}
    

//=====================================================================================
//
// write data methods
//

void LOW_linkPassiveSerial::writeData( const bool inSendBit, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  if ( touchBit( inSendBit, inPullup) != inSendBit )
    throw comm_error( "Response not equal to sent bit", __FILE__, __LINE__);
}


void LOW_linkPassiveSerial::writeData( const uint8_t inSendByte, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  if ( touchByte( inSendByte, inPullup) != inSendByte )
    throw comm_error( "Response not equal to sent byte", __FILE__, __LINE__);
}


void LOW_linkPassiveSerial::writeData( const byteVec_t &inSendBytes, const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  byteVec_t readVec = touchBlock( inSendBytes, inPullup);
  
  for( unsigned int i=0; i<inSendBytes.size(); i++)
    if ( readVec[i] != inSendBytes[i] )
      throw comm_error( "Response not equal to sent byte", __FILE__, __LINE__);
}


//=====================================================================================
//
// misc methods
//

void LOW_linkPassiveSerial::resetLinkAdapter()
{
  commLock lock( *this);
  
  serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit6_size, 
                             LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, LOW_portSerial::B115200_speed);
  
  serialPort->tty_flush();
}

    
bool LOW_linkPassiveSerial::resetBus()
{
  commLock lock( *this);
  
  const uint8_t   resetSendByte = 0xF0;
  
  bool            devFound = false;
    
  try {
    
    // Flush the input and output buffers
    serialPort->tty_flush();
    
    serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit8_size, 
                               LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, LOW_portSerial::B10472_speed);
    
    // Send the reset pulse
    serialPort->tty_write( resetSendByte);

    uint8_t tmpByte = serialPort->tty_readByte();

    if( tmpByte == 0 )  // if answer is still zero, then it is a short to ground
      throw comm_error( "Possible short to ground detected", __FILE__, __LINE__);

    // If the byte read is not what we sent, check for an error
    // For now just return a 1 and discount any errors??
    if( tmpByte != resetSendByte )
      devFound = true; // got a response of some type
    else
      devFound = false; // no device responding
  }
  catch( ... ) {
    
    try { 
      serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit6_size, 
                                 LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, LOW_portSerial::B115200_speed);
    }
    catch( ... ) {}
    
    throw;
  }  
  
  serialPort->tty_configure( LOW_portSerial::none_flowControl, LOW_portSerial::bit6_size, 
                             LOW_portSerial::no_parity, LOW_portSerial::bit1_stopBit, LOW_portSerial::B115200_speed);

  return devFound;
}


void LOW_linkPassiveSerial::strongPullup( const unsigned long /*inMicroSecs*/ )
{ 
  commLock lock( *this);
  
  //LOW_helper_timer::microSleep( inMicroSecs);
  
  LOW_platformMisc::secSleep( strongPullupEmuTime);
}

  
void LOW_linkPassiveSerial::programPulse()
{
  commLock lock( *this);
  
  if ( ! allowProgPulse )
    throw notAllowed_error( "Program pulse not allowed", __FILE__, __LINE__);

  if ( ! hasProgramPulse )
    throw illegalLevel_error( "Program pulse not available", __FILE__, __LINE__);

  throw internal_error( "Program pulse not available, should never be reached", __FILE__, __LINE__);
}

  
void LOW_linkPassiveSerial::doSearchSequence( const LOW_deviceIDRaw &inBranchVector, 
                                              LOW_deviceIDRaw &outFoundID, LOW_deviceIDRaw &outDiscrVec)
{
  commLock lock( *this);
  
  for( int a=0; a<64; a++) {
    
    bool readB = readDataBit();
    bool cmplB = readDataBit();
  
    if ( readB == cmplB ) {
      writeData( inBranchVector.getBit( a));
      outFoundID.setBit( a, inBranchVector.getBit( a));
      outDiscrVec.setBit( a, true);
    }
    else {
      writeData( readB);
      outFoundID.setBit( a, readB);
      outDiscrVec.setBit( a, false);
    }
    
  }
}

    

//=========================================================================================
//
// private methods
//

void LOW_linkPassiveSerial::strongPullup( const strongPullup_t inPullup)
{
  commLock lock( *this);
  
  switch ( inPullup )
  {
    case pullUp_NONE: break;
    case pullUp_16_4: strongPullup(   16400);  break;
    case pullUp_65_5: strongPullup(   65500);  break;
    case pullUp_131:  strongPullup(  131000);  break;
    case pullUp_262:  strongPullup(  262000);  break;
    case pullUp_524:  strongPullup(  524000);  break;
    case pullUp_1048: strongPullup( 1048000);  break;
    default:          throw internal_error( "Unknown strong pullup type", __FILE__, __LINE__);
  }
}

    
